#-------------------------------------------------------------------------------
# Name:        Download_USGS_Watershed_NHD_Data
# Purpose:     Download USGS NHD data zipfile from ftp address online.
#              The zipfiles are watershed-level NHD data that match the watershed the user selects in ArcMap and exports as a shapefile.
#              The zipfile will be downloaded to the directory the user specifies.
#
# Author:      Shannon C.F. Smith
#
# Created:     12/11/2015
# Updates:     6/16/2018 updated FTP files to match USGS server
# Licence:     GPL v3.0
#-------------------------------------------------------------------------------
# Requirements - To use this script you must:
#                   1. Have ArcMap installed on your computer
#                   2. Have a Python IDE installed on your computer
#                   3. Modify the parameters below
#
# Notes:    This script was created to download one zipfile of HU8 and HU4 watershed level data from the ftp address given.
#           Other watershed levels are not located at the given ftp address.
#           This script was tested with PyScripter 2.6 and ArcMap 10.3
#           With modification, this script can be used to download zipfiles from other sites (including http sites).
#           USGS ftp address for reference: ftp://rockyftp.cr.usgs.gov/vdelivery/Datasets/Staged/Hydro/FileGDB101/
#-------------------------------------------------------------------------------

# Main code
import arcpy
import zipfile
import urllib
import os
from zipfile import ZipFile
from urllib import urlretrieve
import arcpy.mapping

# PARAMETERS

# TEST VALUES
# your watershed(s) of interest (select watershed in ArcMap and Export Selected Features as a shapefile)
selected_watersheds_shp = r"C:\Users\barry\Dropbox\Teaching\GIS Programming\Students\F15\ShannonSmith\test\HU8test.shp"
# name of desired output folder
outfolder=r'C:\Users\barry\Dropbox\Teaching\GIS Programming\Students\F15\ShannonSmith\test\out'
# watershed level
level=8

# SCRIPT PARAMETERS
if True:
    selected_watersheds_shp=arcpy.GetParameterAsText(0)
    outfolder=arcpy.GetParameterAsText(1)
    level=arcpy.GetParameter(2)

# get base URL from watershed level
urlbase=r'ftp://rockyftp.cr.usgs.gov/vdelivery/Datasets/Staged/Hydrography/NHD/HU{}/HighResolution/GDB/'.format(str(level))




def reportMsg(msg):
    print msg
    arcpy.AddMessage(msg)

def download_zipfile(outfilename,outfolder,url_address):
    try:
        reportMsg("downloading from {}".format(url_address))
        request = urllib.urlretrieve(url_address, outfilename)
        thefile = ZipFile(outfilename)
    except:
        reportMsg("Error in ftp address - unable to download")
        return      #returns None when an error occurred trying to download zipfile
    else:
        reportMsg("Extracting files from zipfile...")
        thefile.extractall(outfolder)
        thefile.close()
        reportMsg(r"Geodatabase located in {}".format(outfolder))
        # return path to NHDFlowline
        gdbpath=os.path.basename(url_address)
        gdbpath=gdbpath.replace("zip","gdb")
        fcpath = '{}\\{}\\Hydrography\\NHDFlowline'.format(outfolder,gdbpath)
        return fcpath


import arcpy

def addToMap(layerpath):
    # get the map document
    mxd = arcpy.mapping.MapDocument("CURRENT")

    # get the data frame
    df = arcpy.mapping.ListDataFrames(mxd,"*")[0]

    # create a new layer
    newlayer = arcpy.mapping.Layer(layerpath)

    # add the layer to the map at the top of the TOC in data frame 0
    arcpy.mapping.AddLayer(df, newlayer,"TOP")

    # Refresh things
    del mxd, df, newlayer

# run a Search Cursor on the selected watershed shapefile to get the HUC and name of watershed
huc_field="HUC{}".format(level)
sc = arcpy.da.SearchCursor(selected_watersheds_shp,[huc_field,"NAME"])
for row in sc:
    HUC_code = str(row[0])
    huc_name = (row[1])
    reportMsg( "Attempting to download watershed {0} ({1})".format(HUC_code,huc_name))
    # replace " " in watershed name with "_" so url address will be correct
    HUC_name = str(huc_name.replace(" ","_"))
    # print HUC_name
    # format the url address for each watershed in the shapefile
    theurl = "{}NHD_H_{}_HU{}_GDB.zip".format(urlbase,HUC_code,level)
    fcpath =    download_zipfile(HUC_name,outfolder,theurl)
    reportMsg("adding to map:")
    reportMsg(fcpath)
    addToMap(fcpath)
del row
del sc

arcpy.RefreshActiveView()
arcpy.RefreshTOC()

reportMsg("Finished.")

# Modify the following parameter before continuing:
# Copy and paste the url address of the desired watershed zipfile from Interpreter window
# original: url_address = "ftp://rockyftp.cr.usgs.gov/vdelivery/Datasets/Staged/Hydro/FileGDB101/NHD_0903_Rainy_HU4.zip"





